/*
 * gcr.c
 * GCR encoding/decoding utility
 *
 */

#include "gcr.h"

/*
 * raw track
 * this is obtained by
 * 14.31818MHz / 14 / 32 / 8
 *
 */
#define RAW_TRACK_BYTES TRACK_SIZE
#define DOS_TRACK_BYTES 4096 
#define RAW_TRACK_BITS (RAW_TRACK_BYTES*8) 

const byte	GCR_encoding_table[64] = {
	0x96, 0x97, 0x9A, 0x9B, 0x9D, 0x9E, 0x9F, 0xA6,
	0xA7, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB2, 0xB3,
	0xB4, 0xB5, 0xB6, 0xB7, 0xB9, 0xBA, 0xBB, 0xBC,
	0xBD, 0xBE, 0xBF, 0xCB, 0xCD, 0xCE, 0xCF, 0xD3,
	0xD6, 0xD7, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE,
	0xDF, 0xE5, 0xE6, 0xE7, 0xE9, 0xEA, 0xEB, 0xEC,
	0xED, 0xEE, 0xEF, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6,
	0xF7, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF };

/* physical sector no. to DOS 3.3 logical sector no. table */
const byte	Logical_Sector[16] = {
	0x0, 0x7, 0xE, 0x6, 0xD, 0x5, 0xC, 0x4,
	0xB, 0x3, 0xA, 0x2, 0x9, 0x1, 0x8, 0xF };

const byte	Swap_Bit[4] = { 0, 2, 1, 3 }; /* swap lower 2 bits */

TAppleGCRCodec::TAppleGCRCodec()
{
	Position = 0;
	init_GCR_table();
}

void TAppleGCRCodec::init_GCR_table(void)
{
	int		i;

	{	
	   for( i = 0; i < 64; i++ )
	      GCR_decoding_table[GCR_encoding_table[i]] = (byte)i;
	   for( i = 0; i < 16; i++ )
	      Physical_Sector[Logical_Sector[i]] = (byte)i;
	}
}

inline byte TAppleGCRCodec::gcr_read_nibble(void)
{ 
	byte	data;
	
	data = Track_Nibble[Position++];
	if ( Position >= RAW_TRACK_BYTES )
	   Position = 0;	
	return data;
}

inline void TAppleGCRCodec::gcr_write_nibble( byte data )
{
	Track_Nibble[Position++] = data;
	if ( Position >= RAW_TRACK_BYTES )
	   Position = 0;
}

void TAppleGCRCodec::decode62( byte *page )
{
	int	i, j;

	/* get 6 bits from GCR_buffer & 2 from GCR_buffer2 */
	for( i = 0, j = 86; i < 256; i++ ) {
	  if ( --j < 0 ) j = 85;
	  page[i] = (GCR_buffer[i] << 2) | Swap_Bit[GCR_buffer2[j] & 0x03];
	  GCR_buffer2[j] >>= 2;
	}
}

void TAppleGCRCodec::encode62( byte *page )
{
	int	i, j;

	/* 86 * 3 = 258, so the first two byte are encoded twice */
	GCR_buffer2[0] = Swap_Bit[page[1]&0x03];
	GCR_buffer2[1] = Swap_Bit[page[0]&0x03];

	/* save higher 6 bits in GCR_buffer and lower 2 bits in GCR_buffer2 */
	for( i = 255, j = 2; i >= 0; i--, j = j == 85? 0: j + 1 ) {
	   GCR_buffer2[j] = (GCR_buffer2[j] << 2) | Swap_Bit[page[i]&0x03];
	   GCR_buffer[i] = page[i] >> 2;
	}
	 
	/* clear off higher 2 bits of GCR_buffer2 set in the last call */
	for( i = 0; i < 86; i++ )
	   GCR_buffer2[i] &= 0x3f;
}

/*
 * write an FM encoded value, used in writing address fields 
 */
void TAppleGCRCodec::FM_encode( byte data )
{
	gcr_write_nibble( (data >> 1) | 0xAA );
	gcr_write_nibble( data | 0xAA );

}

/*
 * return an FM encoded value, used in reading address fields 
 */
byte TAppleGCRCodec::FM_decode(void)
{
	int		tmp; 

	/* C does not specify order of operand evaluation, don't
	 * merge the following two expression into one
	 */
	tmp = (gcr_read_nibble() << 1) | 0x01;
	return gcr_read_nibble() & tmp;
}

void TAppleGCRCodec::write_sync( int length )
{
	while( length-- )
	   gcr_write_nibble( 0xFF );
}

/*
 * read_address_field: try to read a address field in a track
 * returns 1 if succeed, 0 otherwise
 */
int TAppleGCRCodec::read_address_field( int *volume, int *track, int *sector )
{
	int	max_try;
	byte	nibble;

	max_try = 100; 
	while( --max_try ) {
	   nibble = gcr_read_nibble(); 
	   check_D5:	
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	   goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0x96 )
	      goto check_D5;
	   *volume = FM_decode();
	   *track = FM_decode();
	   *sector = FM_decode();
	   return ( *volume ^ *track ^ *sector ) == FM_decode() &&
	      gcr_read_nibble() == 0xDE;
	}
	return 0;
}

void TAppleGCRCodec::write_address_field( int volume, int track, int sector )
{
	/*
	 * write address mark
	 */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0x96 );

	/*
	 * write Volume, Track, Sector & Check-sum
	 */
	FM_encode( volume );
	FM_encode( track );
	FM_encode( sector );
	FM_encode( volume ^ track ^ sector );

	/*
	 * write epilogue
	 */
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

/*
 * read_data_field: read_data_field into GCR_buffers, return 0 if fail
 */
int TAppleGCRCodec::read_data_field(void)
{
	int	i, max_try;
	byte	nibble, checksum;

	/*
	 * read data mark
	 */
	max_try = 32;
	while( --max_try ) {
	   nibble = gcr_read_nibble();
	check_D5:
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	      goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble == 0xAD )
	      break;
	}
	if ( !max_try ) /* fails to get address mark */
	   return 0;

 	for( i = 0x55, checksum = 0; i >= 0; i-- ) {
	   checksum ^= GCR_decoding_table[gcr_read_nibble()]; 
	   GCR_buffer2[i] = checksum;
	}

	for( i = 0; i < 256; i++ ) {
	   checksum ^= GCR_decoding_table[gcr_read_nibble()]; 
	   GCR_buffer[i] = checksum;
	}

	/* verify sector checksum */
	if ( checksum ^ GCR_decoding_table[gcr_read_nibble()] )
	   return 0; 

	/* check epilogue */
	return gcr_read_nibble() == 0xDE && gcr_read_nibble() == 0xAA;
}

void TAppleGCRCodec::write_data_field(void)
{
	int	i;
	byte	last, checksum;

	/* write prologue */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xAD );

	/* write GCR encode data */
 	for( i = 0x55, last = 0; i >= 0; i-- ) {
	   checksum = last^ GCR_buffer2[i];
	   gcr_write_nibble( GCR_encoding_table[checksum] );
	   last = GCR_buffer2[i];
	}
	for( i = 0; i < 256; i++ ) {
	   checksum = last ^ GCR_buffer[i];
	   gcr_write_nibble( GCR_encoding_table[checksum] );
	   last = GCR_buffer[i];
	}

	/* write checksum and epilogue */
	gcr_write_nibble( GCR_encoding_table[last] );
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

void TAppleGCRCodec::nibblizeTrack(int volume, int track, 
	TAppleCookedTrack in, TAppleRawTrack out)
{
	int	i;

	//init_GCR_table();
	Track_Nibble = out;
	Position = 0;

	//write_sync( 48 );
	for( i = 0; i < 16; i ++ ) {
	   encode62( in + ((int)Logical_Sector[i]) * 0x100 );
	   write_sync( 14 /*27*/ );
	   write_address_field( volume, track, i );
	   write_sync( 6 );
	   write_data_field();
	}
	write_sync(RAW_TRACK_BYTES-Position);
}

int TAppleGCRCodec::denibblizeTrack(int volume, int track, 
	TAppleRawTrack in, TAppleCookedTrack out)
{
	int scanned;
	int	i, max_try;
	int	vv, tt, ss;	/* volume, track no. and sector no. */

	Track_Nibble = in;
	Position = 0;

	scanned = 0;

	max_try = 200;
	while( --max_try ) { 
	   if ( !read_address_field( &vv, &tt, &ss ) )
	      continue;

	   if ( (volume && vv != volume ) || tt != track || ss < 0 || ss > 15 ){
	      //printf("phy sector %d address field invalid\n", ss );
	      continue;	/* invalid values for vv, tt and ss, try again */
	   }

	   ss = Logical_Sector[ss];
	   if ( scanned & (1<<ss) )	/* sector has been read */
	      continue;

	   if ( read_data_field() ) {
	      decode62( out + ss * 0x100 );
	      scanned |= (1<<ss);	/* this sector's ok */ 
	   }
	   else {
	      //printf("fail reading data field of logical sector %d\n", ss );
	   }
	}

	/* if has failed to read any one sector, report error */
	return (scanned == 0xffff);
}

